extends Node2D
# Este script controla un minijuego muy simple:
# - Una nave (el jugador) se mueve a izquierda/derecha en la parte inferior.
# - Aparecen asteroides desde arriba.
# - Si un asteroide choca con la nave: GAME OVER.
#
# ARQUITECTURA EN GODOT:
# - _ready() se ejecuta una vez al iniciar la escena (inicializaciones).
# - _process(delta) se ejecuta cada frame (lógica del juego “por frames”).
# - _draw() se ejecuta cuando se redibuja el nodo (dibujo 2D inmediato).
# - _input(event) recibe los eventos de entrada (teclado, ratón, táctil…).
#
# IMPORTANTE:
# - Mapea las acciones en Project → Project Settings → Input Map:
#   "ui_left", "ui_right" y "ui_accept".
# - Coloca las imágenes en las rutas indicadas (res://jugador.png, etc.).
# - Este ejemplo usa “dibujo inmediato” (draw_* en _draw) en lugar de nodos Sprite.

# -------------------------
# CONSTANTES DE CONFIGURACIÓN
# -------------------------

# Tamaños de jugador, asteroides, etc.
const TAM_JUGADOR = 128
const TAM_ASTEROIDE = 64
const FACTOR_HITBOX = 0.75   # Factor para reducir un poco las "cajas de choque" (colisiones menos “injustas”)
const TAM_BOTON_CERRAR = 32  # Tamaño del botón para cerrar el juego (arriba derecha)
const TAM_TEXTO = 25         # Tamaño del texto de las etiquetas (UI)

# Colores que se usan en el juego (Color(r, g, b, a) con valores 0..1)
const COLOR_FONDO = Color(0, 0, 0)                 # Fondo negro cuando jugamos
const COLOR_GAME_OVER = Color(0.2, 0.0, 0.0, 0.5)  # Capa roja semitransparente al perder (efecto “oscurecido”)

# Imágenes (texturas) del jugador, del asteroide y del botón cerrar.
# preload() carga el recurso al iniciar el juego (más eficiente que load() en tiempo de ejecución)
const TEX_JUGADOR = preload("res://jugador.png")
const TEX_ASTEROIDE = preload("res://asteroide.png")
const TEX_BOTON_CERRAR = preload("res://boton_cerrar.png")

# Velocidades y tiempos (valores “jugables”, se pueden tunear)
var vel_jugador = 500.0           # Pixels/segundo que se moverá el jugador
var vel_asteroides = 250.0        # Velocidad vertical de los asteroides (pixels/segundo)
var intervalo_asteroides = 0.75   # Cada cuánto aparece un nuevo asteroide (en segundos)

# -------------------------
# ESTADO DEL JUEGO
# -------------------------

# “Rect2” define posición y tamaño (position.x, position.y, size.x, size.y)
var boton_cerrar: Rect2
var jugador: Rect2
var asteroides: Array[Rect2] = []    # Lista de rectángulos de asteroides

# Variables de control
var tiempo_proximo_asteroide = 0.0   # Acumula el tiempo hasta crear el siguiente asteroide
var muerto = false                   # Marca si ya hemos perdido (pausa la lógica de juego)
var pantalla: Vector2                # Tamaño de la ventana/viewport (ancho, alto)
var tiempo_total = 0.0               # Cronómetro de la partida (en segundos)

# Etiquetas de texto (UI)
var etiqueta_game_over: Label
var etiqueta_tiempo: Label

# Controles del jugador (banderas para izquierda/derecha)
var tocando_izquierda = false
var tocando_derecha = false

# -------------------------
# CICLO DE VIDA PRINCIPAL
# -------------------------

func _ready():
	# Se llama una vez al cargar la escena. Preparamos todo.
	randomize() # Se utiliza una "semilla" distinta en cada partida para generar números aleatorios diferentes
	_inicializar_pantalla()
	_inicializar_jugador()
	_crear_ui_tiempo()
	_crear_boton_cerrar()
	_crear_ui_game_over()

func _process(delta: float):
	# delta = tiempo (en segundos) entre este frame y el anterior.
	# Si el jugador está muerto, detenemos la lógica (pero se puede seguir dibujando).
	if muerto: return

	# Orden típico de actualización por frame:
	_mover_jugador(delta)
	_crear_asteroides(delta)
	_mover_asteroides(delta)
	_comprobar_colision()
	_actualizar_tiempo(delta)
	_actualizar_dificultad(delta)

	# Pedimos que se llame a _draw() para redibujar (importante para ver cambios visuales).
	queue_redraw()

func _draw():
	# Todo lo que dibujamos aquí se "pinta" encima del lienzo del Node2D en este frame.
	_color_fondo_jugando()  # Capa de fondo (negro)
	_dibujar_asteroides()   # Asteroides (texturas)
	_dibujar_jugador()      # Nave del jugador (textura)
	_mostrar_boton_cerrar() # Botón "X" arriba a la derecha
	if muerto: _color_game_over()  # Capa roja semitransparente al perder (se dibuja al final)

func _input(event: InputEvent):
	# _input recibe eventos de teclado, ratón y táctiles tal cual ocurren.
	# Los separamos en dos funciones para mantener limpio el código.
	_comprobar_pantalla_tactil_y_raton(event)
	_comprobar_teclado(event)

# -------------------------
# FUNCIONES DE INICIO
# -------------------------

func _inicializar_pantalla():
	# Guardamos el tamaño actual de la pantalla/viewport (Vector2(ancho, alto)).
	pantalla = get_viewport_rect().size

func _inicializar_jugador():
	# Colocamos al jugador centrado horizontalmente y un poco por encima del borde inferior.
	jugador = Rect2(
		pantalla.x / 2 - TAM_JUGADOR / 2,     # x (centrado)
		pantalla.y - TAM_JUGADOR * 1.25,      # y (cerca del borde inferior)
		TAM_JUGADOR,                          # ancho
		TAM_JUGADOR                           # alto
	)

func _dibujar_jugador():
	# Dibujamos la nave del jugador usando su rectángulo como destino.
	draw_texture_rect(TEX_JUGADOR, jugador, false)

# -------------------------
# CONTROLES
# -------------------------

func _comprobar_pantalla_tactil_y_raton(event: InputEvent):
	# Control con pantalla táctil o ratón:
	# - Si pulsas sobre el botón de cerrar: se termina el juego.
	# - Si pulsas a la izquierda de la nave: mueves a la izquierda (y al revés).
	# - Al soltar: se detiene el movimiento.
	if event is InputEventScreenTouch or event is InputEventMouseButton:
		if event.pressed:
			# Si se ha pulsado el botón de cerrar el juego, finalizamos la ejecución de la aplicación de forma segura.
			if boton_cerrar.has_point(event.position):
				if get_tree(): 
					get_tree().quit.call_deferred()
				return
			
			# Decidimos la dirección según dónde has pulsado respecto al centro de la nave.
			var centro_nave = jugador.position.x + TAM_JUGADOR / 2
			if event.position.x < centro_nave:
				tocando_izquierda = true
				tocando_derecha = false
			else:
				tocando_derecha = true
				tocando_izquierda = false
		else:
			# Al soltar botón o dejar de tocar la pantalla, detenemos el movimiento.
			tocando_izquierda = false
			tocando_derecha = false

func _comprobar_teclado(event: InputEvent):
	# Reiniciar el juego si hemos perdido: “ui_accept” suele ser Enter o Espacio.
	if muerto and event.is_action_pressed("ui_accept"):	
		_reiniciar_juego()
		return

	# Controles con teclado. Usamos acciones abstractas (Input Map) en lugar de teclas fijas.
	if event.is_action_pressed("ui_left"):
		tocando_izquierda = true
	if event.is_action_released("ui_left"):
		tocando_izquierda = false

	if event.is_action_pressed("ui_right"):
		tocando_derecha = true
	if event.is_action_released("ui_right"):
		tocando_derecha = false

# -------------------------
# MOVIMIENTO DEL JUGADOR
# -------------------------

func _mover_jugador(delta: float):
	var dir = 0  # Dirección horizontal: -1 izquierda, 0 quieto, +1 derecha
	
	# Revisamos qué bandera de control está activa.
	if tocando_izquierda: dir = -1
	elif tocando_derecha: dir = +1

	# Sumamos desplazamiento = velocidad * tiempo * dirección.
	# clamp evita que el jugador se salga por los bordes (0 a pantalla.x - tamaño).
	jugador.position.x = clamp(
		jugador.position.x + dir * vel_jugador * delta,
		0, pantalla.x - TAM_JUGADOR
	)

# -------------------------
# BOTÓN PARA CERRAR LA APLICACIÓN
# -------------------------

func _crear_boton_cerrar():
	# Creamos un rectángulo en la esquina superior derecha con el tamaño indicado.
	boton_cerrar = Rect2(pantalla.x - TAM_BOTON_CERRAR, 0, TAM_BOTON_CERRAR, TAM_BOTON_CERRAR)

func _mostrar_boton_cerrar():
	# Dibujamos el icono del botón de cerrar. Al hacer clic/tap encima, cerramos el juego.
	draw_texture_rect(TEX_BOTON_CERRAR, boton_cerrar, false)	

# -------------------------
# INTERFAZ DE USUARIO
# -------------------------  

func _crear_ui_tiempo():
	# Crea la etiqueta que muestra el tiempo de supervivencia.
	# Por defecto la Label aparece en (0, 0), es decir, en la esquina superior izquierda
	etiqueta_tiempo = Label.new()
	etiqueta_tiempo.text = "0.0 s"
	etiqueta_tiempo.set("theme_override_font_sizes/font_size", TAM_TEXTO)
	etiqueta_tiempo.visible = true
	add_child(etiqueta_tiempo)

func _actualizar_tiempo(delta: float):
	# Si seguimos vivos, sumamos delta al cronómetro.
	if not muerto:
		tiempo_total += delta
	# snappedf(x, step) redondea con un “paso”. Aquí dejamos 1 decimal.
	etiqueta_tiempo.text = str(snappedf(tiempo_total, 0.1)) + " s"

func _crear_ui_game_over():
	# Crea la etiqueta de "GAME OVER" (oculta hasta que perdamos).
	etiqueta_game_over = Label.new()
	etiqueta_game_over.set("theme_override_font_sizes/font_size", TAM_TEXTO)
	etiqueta_game_over.visible = false
	add_child(etiqueta_game_over)

# -------------------------
# ASTEROIDES
# -------------------------

func _crear_asteroides(delta: float):
	# Cada “intervalo_asteroides” segundos, creamos un asteroide nuevo arriba.
	tiempo_proximo_asteroide += delta
	if tiempo_proximo_asteroide >= intervalo_asteroides:
		tiempo_proximo_asteroide = 0.0

		# Posición X aleatoria dentro de los límites de la pantalla.
		var x = randi_range(0, pantalla.x - TAM_ASTEROIDE)

		# Tamaño aleatorio entre la mitad y el doble del tamaño base.
		var tam = randi_range(TAM_ASTEROIDE / 2.0, TAM_ASTEROIDE * 2.0)

		# Colocamos el rectángulo justo por encima de la pantalla (y negativo), y así “entra” cayendo.
		var asteroide = Rect2(x, -tam, tam, tam)
		asteroides.append(asteroide)

func _mover_asteroides(delta: float):
	# Los asteroides caen hacia abajo a velocidad constante (vel_asteroides).
	for i in asteroides.size():
		asteroides[i].position.y += vel_asteroides * delta

	# Eliminamos los que ya salieron de la pantalla (su y es mayor que la altura).
	# filter crea una nueva lista con los elementos que cumplan la condición.
	asteroides = asteroides.filter(func(o): return o.position.y < pantalla.y)

func _dibujar_asteroides():
	# Dibujamos cada asteroide con su textura y su rectángulo correspondiente.
	for asteroide in asteroides:
		draw_texture_rect(TEX_ASTEROIDE, asteroide, false)

# -------------------------
# COLISIONES Y GAME OVER
# -------------------------

func _game_over():
	# Marcamos estado de derrota y actualizamos UI.
	muerto = true
	tocando_izquierda = false
	tocando_derecha = false	
	etiqueta_game_over.text = "GAME OVER (" + etiqueta_tiempo.text + ")"
	etiqueta_tiempo.visible = false
	etiqueta_game_over.visible = true

func _reiniciar_juego():
	# Recargamos la escena actual (vuelta a empezar).
	# Todas las variables vuelven a su valor inicial definido en el script.
	get_tree().reload_current_scene()

func _escalar_rect(r: Rect2):
	# Hacemos la “hitbox” un poco más pequeña que el sprite real para
	# que las colisiones estén más ajustadas.
	# Rect2.grow(margen) crece (margen > 0) o encoge (margen < 0) por todos los lados.
	# Aquí calculamos un margen negativo proporcional al ancho.
	return r.grow((FACTOR_HITBOX - 1.0) * r.size.x / 2)

func _comprobar_colision():
	# Comprobamos si el rect del jugador (reducido) intersecta con alguno de los asteroides.
	var jugador_escalado = _escalar_rect(jugador)
	for asteroide in asteroides:
		var asteroide_escalado = _escalar_rect(asteroide)
		if jugador_escalado.intersects(asteroide_escalado):
			_game_over()

# -------------------------
# DIBUJO DE FONDOS
# -------------------------

func _color_fondo_jugando():
	# Pintamos un rectángulo del tamaño de la pantalla con color de fondo.
	draw_rect(Rect2(Vector2.ZERO, pantalla), COLOR_FONDO, true)
		
func _color_game_over():
	# Dibujamos una capa roja semitransparente por encima al perder.
	# Truco visual: se ve el juego “oscurecido” debajo.
	draw_rect(Rect2(Vector2.ZERO, pantalla), COLOR_GAME_OVER, true)

# -------------------------
# DIFICULTAD PROGRESIVA
# -------------------------

func _actualizar_dificultad(delta: float):
	# Aumenta poco a poco la velocidad de caída de los asteroides
	vel_asteroides += 10 * delta
	
	# Disminuye el tiempo entre asteroides (aparecen más rápido)
	# Usamos max() para poner un límite inferior y que nunca baje de 0.1 s.
	intervalo_asteroides = max(0.1, intervalo_asteroides - 0.01 * delta)
